import React, { createContext, useContext, useState, useCallback, useEffect } from 'react';
import { ADMIN_CREDENTIALS } from '@/data/users';
import { loginAPI, getCredentials, updateCredentials, setAdminCredentials, clearAdminCredentials } from '@/services/api';
import type { User } from '@/types';

interface CustomerCredentials {
  username: string;
  password: string;
}

interface AuthContextType {
  isAuthenticated: boolean;
  user: User | null;
  role: 'user' | 'admin' | null;
  customerCredentials: CustomerCredentials;
  isLoading: boolean;
  error: string | null;
  login: (username: string, password: string) => Promise<boolean>;
  logout: () => void;
  updateCustomerCredentials: (username: string, password: string) => Promise<void>;
  refreshCredentials: () => Promise<void>;
}

const AuthContext = createContext<AuthContextType | undefined>(undefined);

// Default credentials
const DEFAULT_CREDENTIALS: CustomerCredentials = {
  username: 'customer',
  password: 'screen2024',
};

export function AuthProvider({ children }: { children: React.ReactNode }) {
  const [isAuthenticated, setIsAuthenticated] = useState(false);
  const [user, setUser] = useState<User | null>(null);
  const [role, setRole] = useState<'user' | 'admin' | null>(null);
  const [customerCredentials, setCustomerCredentials] = useState<CustomerCredentials>(DEFAULT_CREDENTIALS);
  const [isLoading, setIsLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);

  // Load credentials from API on mount
  useEffect(() => {
    refreshCredentials();
  }, []);

  const refreshCredentials = useCallback(async () => {
    try {
      setIsLoading(true);
      const response = await getCredentials();
      if (response.success && response.data) {
        setCustomerCredentials(response.data);
      }
    } catch (err) {
      console.error('Failed to fetch credentials:', err);
      setError('Failed to connect to server');
    } finally {
      setIsLoading(false);
    }
  }, []);

  const login = useCallback(async (username: string, password: string): Promise<boolean> => {
    setError(null);
    
    try {
      // Check admin credentials locally first (for speed)
      if (username === ADMIN_CREDENTIALS.username && password === ADMIN_CREDENTIALS.password) {
        setIsAuthenticated(true);
        setUser({ username, password, role: 'admin' });
        setRole('admin');
        setAdminCredentials(username, password);
        return true;
      }
      
      // Try API login for customer
      const response = await loginAPI(username, password);
      
      if (response.success) {
        setIsAuthenticated(true);
        setUser({ username, password, role: response.role });
        setRole(response.role);
        
        if (response.role === 'admin') {
          setAdminCredentials(username, password);
        }
        
        return true;
      }
      
      return false;
    } catch (err) {
      console.error('Login error:', err);
      setError('Login failed. Please check your connection.');
      return false;
    }
  }, []);

  const logout = useCallback(() => {
    setIsAuthenticated(false);
    setUser(null);
    setRole(null);
    clearAdminCredentials();
  }, []);

  const updateCustomerCredentials = useCallback(async (username: string, password: string) => {
    try {
      const response = await updateCredentials(username, password);
      if (response.success) {
        setCustomerCredentials({ username, password });
      }
    } catch (err) {
      console.error('Failed to update credentials:', err);
      throw err;
    }
  }, []);

  return (
    <AuthContext.Provider
      value={{ 
        isAuthenticated, 
        user, 
        role, 
        customerCredentials, 
        isLoading,
        error,
        login, 
        logout, 
        updateCustomerCredentials,
        refreshCredentials
      }}
    >
      {children}
    </AuthContext.Provider>
  );
}

export function useAuth() {
  const context = useContext(AuthContext);
  if (context === undefined) {
    throw new Error('useAuth must be used within an AuthProvider');
  }
  return context;
}
