import React, { createContext, useContext, useState, useCallback, useEffect } from 'react';
import { getProducts, addProductAPI, updateProductAPI, deleteProductAPI } from '@/services/api';
import type { Product } from '@/types';

interface InventoryContextType {
  products: Product[];
  isLoading: boolean;
  error: string | null;
  addProduct: (product: Omit<Product, 'id'>) => Promise<void>;
  updateProduct: (id: string, updates: Partial<Product>) => Promise<void>;
  deleteProduct: (id: string) => Promise<void>;
  getCategories: () => string[];
  refreshProducts: () => Promise<void>;
}

const InventoryContext = createContext<InventoryContextType | undefined>(undefined);

export function InventoryProvider({ children }: { children: React.ReactNode }) {
  const [products, setProducts] = useState<Product[]>([]);
  const [isLoading, setIsLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);

  // Load products from API on mount
  useEffect(() => {
    refreshProducts();
  }, []);

  const refreshProducts = useCallback(async () => {
    try {
      setIsLoading(true);
      setError(null);
      const response = await getProducts();
      if (response.success && response.data) {
        setProducts(response.data);
      }
    } catch (err) {
      console.error('Failed to fetch products:', err);
      setError('Failed to load products');
    } finally {
      setIsLoading(false);
    }
  }, []);

  const addProduct = useCallback(async (product: Omit<Product, 'id'>) => {
    try {
      const newProduct: Product = {
        ...product,
        id: `prod_${Date.now()}`,
      };
      
      await addProductAPI(newProduct);
      
      // Refresh products from server
      await refreshProducts();
    } catch (err) {
      console.error('Failed to add product:', err);
      throw err;
    }
  }, [refreshProducts]);

  const updateProduct = useCallback(async (id: string, updates: Partial<Product>) => {
    try {
      const product = products.find(p => p.id === id);
      if (!product) throw new Error('Product not found');
      
      const updatedProduct = {
        category: updates.category || product.category,
        model: updates.model || product.model,
        price: updates.price !== undefined ? updates.price : product.price,
      };
      
      await updateProductAPI(id, updatedProduct);
      
      // Refresh products from server
      await refreshProducts();
    } catch (err) {
      console.error('Failed to update product:', err);
      throw err;
    }
  }, [products, refreshProducts]);

  const deleteProduct = useCallback(async (id: string) => {
    try {
      await deleteProductAPI(id);
      
      // Refresh products from server
      await refreshProducts();
    } catch (err) {
      console.error('Failed to delete product:', err);
      throw err;
    }
  }, [refreshProducts]);

  const getCategories = useCallback(() => {
    const categories = new Set(products.map((p) => p.category));
    return Array.from(categories).sort();
  }, [products]);

  return (
    <InventoryContext.Provider
      value={{ 
        products, 
        isLoading, 
        error, 
        addProduct, 
        updateProduct, 
        deleteProduct, 
        getCategories,
        refreshProducts
      }}
    >
      {children}
    </InventoryContext.Provider>
  );
}

export function useInventory() {
  const context = useContext(InventoryContext);
  if (context === undefined) {
    throw new Error('useInventory must be used within an InventoryProvider');
  }
  return context;
}
